<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Faq Widget.
 *
 * @since 1.0
 */

class Tastyc_Faq_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-faq';
	}

	public function get_title() {
		return esc_html__( 'Faq', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'       => esc_html__( 'Subtitle', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter subtitle', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Subtitle', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Title', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => [
					'h1'  => __( 'H1', 'tastyc-plugin' ),
					'h2' => __( 'H2', 'tastyc-plugin' ),
					'h3' => __( 'H3', 'tastyc-plugin' ),
					'h4' => __( 'H4', 'tastyc-plugin' ),
					'div' => __( 'DIV', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'description',
			[
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter description', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Description', 'tastyc-plugin' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'title', [
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default' => esc_html__( 'Enter title', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'description', [
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Enter text', 'tastyc-plugin' ),
				'default' => esc_html__( 'Enter text', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ title }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'     => esc_html__( 'Subtitle Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-suptitle' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
				'label'     => esc_html__( 'Subtitle Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-suptitle',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-title--h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-title--h',
			]
		);

		$this->add_control(
			'desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-title--desc' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-title--desc',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label' => esc_html__( 'Description', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_title_color',
			[
				'label' => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-faq-item-box .name' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'item_title_typography',
				'label' => esc_html__( 'Title Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-faq-item-box .name',
			]
		);

		$this->add_control(
			'item_description_color',
			[
				'label' => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-faq-item-box .text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'item_description_typography',
				'label' => esc_html__( 'Description Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-faq-item-box .text',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'subtitle', 'basic' );
		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'description', 'basic' );

		?>

		<!-- container -->
		<div class="container">

		<div class="row tst-mb-60">
		  <?php if ( $settings['title'] || $settings['subtitle'] || $settings['description'] ) : ?>
          <div class="col-lg-12">

            <!-- title -->
            <div class="text-center">
              <?php if ( $settings['subtitle'] ) : ?>
              <div class="tst-suptitle tst-suptitle-center tst-mb-15">
              	<span <?php echo $this->get_render_attribute_string( 'subtitle' ); ?>>
		          	<?php echo wp_kses_post( $settings['subtitle'] ); ?>
		        </span>
              </div>
              <?php endif; ?>
              <?php if ( $settings['title'] ) : ?>
              <<?php echo esc_attr( $settings['title_tag'] ); ?> class="tst-title--h tst-mb-30">
              	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
		          	<?php echo wp_kses_post( $settings['title'] ); ?>
		        </span>
              </<?php echo esc_attr( $settings['title_tag'] ); ?>>
              <?php endif; ?>
              <?php if ( $settings['description'] ) : ?>
              <p class="tst-text tst-title--desc tst-mb-60">
              	<span <?php echo $this->get_render_attribute_string( 'description' ); ?>>
		          	<?php echo wp_kses_post( $settings['description'] ); ?>
		        </span>
              </p>
              <?php endif; ?>
            </div>
            <!-- title end -->

          </div>
          <?php endif; ?>

          <?php if ( $settings['items'] ) : ?>
          <div class="col-lg-12 faq-items-box collapse-items">
		  <?php foreach ( $settings['items'] as $index => $item ) :
	    	$item_title = $this->get_repeater_setting_key( 'title', 'items', $index );
	    	$this->add_inline_editing_attributes( $item_title, 'basic' );

	    	$item_description = $this->get_repeater_setting_key( 'description', 'items', $index );
	    	$this->add_inline_editing_attributes( $item_description, 'advanced' );
	      ?>

            <!-- services box -->
            <div class="tst-faq-item-box collapse-item">
				<?php if ( $item['title'] ) : ?>
				<div class="name collapse-btn">
					<span <?php echo $this->get_render_attribute_string( $item_title ); ?>>
						<?php echo wp_kses_post( $item['title'] ); ?>
					</span>
				</div>
				<?php endif; ?>
				<?php if ( $item['description'] ) : ?>
				<div class="text collapse-content" style="display: none;">
					<div <?php echo $this->get_render_attribute_string( $item_description ); ?>>
						<?php echo wp_kses_post( $item['description'] ); ?>
					</div>
				</div>
				<?php endif; ?>
            </div>
            <!-- services box end -->

          <?php endforeach; ?>
      	  </div>
          <?php endif; ?>

        </div>

        <div class="tst-spacer tst-spacer-only-bottom-space"></div>

		</div>
		<!-- container end -->

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() { ?>


		<#
		view.addInlineEditingAttributes( 'subtitle', 'basic' );
		view.addInlineEditingAttributes( 'title', 'basic' );
		view.addInlineEditingAttributes( 'description', 'basic' );
		#>

		<!-- container -->
		<div class="container">

		<div class="row">
		  <# if ( settings.title || settings.subtitle || settings.description ) { #>
          <div class="col-lg-12">

            <!-- title -->
            <div class="text-center">
              <# if ( settings.subtitle ) { #>
              <div class="tst-suptitle tst-suptitle-center tst-mb-15">
              	<span {{{ view.getRenderAttributeString( 'subtitle' ) }}}>
		          	{{{ settings.subtitle }}}
		        </span>
              </div>
              <# } #>
              <# if ( settings.title ) { #>
              <{{{ settings.title_tag }}} class="tst-title--h tst-mb-30">
              	<span {{{ view.getRenderAttributeString( 'title' ) }}}>
		          	{{{ settings.title }}}
		        </span>
              </{{{ settings.title_tag }}}>
              <# } #>
              <# if ( settings.description ) { #>
              <p class="tst-text tst-title--desc tst-mb-60">
              	<span {{{ view.getRenderAttributeString( 'description' ) }}}>
		          	{{{ settings.description }}}
		        </span>
              </p>
              <# } #>
            </div>
            <!-- title end -->

          </div>
          <# } #>

          <# if ( settings.items ) { #>
          <div class="col-lg-12 faq-items-box collapse-items">
		  <# _.each( settings.items, function( item, index ) {
				var item_title = view.getRepeaterSettingKey( 'title', 'items', index );
			    view.addInlineEditingAttributes( item_title, 'basic' );

			    var item_description = view.getRepeaterSettingKey( 'description', 'items', index );
			    view.addInlineEditingAttributes( item_description, 'advanced' );
		  #>
          <div class="col-lg-12">

            <!-- services box -->
            <div class="tst-faq-item-box collapse-item">
                <# if ( item.title ) { #>
				<div class="name collapse-btn">
					<span {{{ view.getRenderAttributeString( item_title ) }}}>
						{{{ item.title }}}
					</span>
				</div>
				<# } #>
				<# if ( item.description ) { #>
				<div class="text collapse-content" style="display: none;">
					<div {{{ view.getRenderAttributeString( item_description ) }}}>
						{{{ item.description }}}
					</div>
				</div>
				<# } #>
            </div>
            <!-- services box end -->

          </div>
          <# }); #>
          </div>
          <# } #>

        </div>

        <div class="tst-spacer tst-spacer-only-bottom-space"></div>

		</div>
		<!-- container end -->

	<?php }
}

Plugin::instance()->widgets_manager->register( new Tastyc_Faq_Widget() );
