<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Footer Gallery Widget.
 *
 * @since 1.0
 */
class Tastyc_Footer_Gallery_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-footer-gallery';
	}

	public function get_title() {
		return esc_html__( 'Footer Gallery', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Title', 'tastyc-plugin' ),
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image', [
				'label' => esc_html__( 'Image', 'tastyc-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'name', [
				'label'       => esc_html__( 'Name', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter name', 'tastyc-plugin' ),
				'default'	=> esc_html__( 'Enter name', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ name }}}',
			]
		);

		$this->add_control(
			'link_text',
			[
				'label'       => esc_html__( 'Link Text', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter your link text', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Read More', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'link',
			[
				'label' => esc_html__( 'Link URL', 'tastyc-plugin' ),
				'label_block' => true,
				'type' => Controls_Manager::URL,
				'show_external' => false,
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label' => __( 'Alignment', 'tastyc-plugin' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', 'tastyc-plugin' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'tastyc-plugin' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'tastyc-plugin' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}}' => 'text-align: {{VALUE}};',
				],
				'default'	=> 'left',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_styling',
			[
				'label' => esc_html__( 'Content Styles', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-mb-60 h5' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'label' => esc_html__( 'Title Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-mb-60 h5',
			]
		);

		$this->add_control(
			'link_color',
			[
				'label' => esc_html__( 'Link Color', 'tastyc-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .tst-mb-60 .tst-label' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'link_typography',
				'label' => esc_html__( 'Link Typography:', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-mb-60 .tst-label',
			]
		);

		$this->end_controls_section();

	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		?>

		<div class="tst-mb-60">
			<?php if ( $settings['title'] ) : ?>
        	<h5 class="tst-mb-30 tst-text-shadow">
        		<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
					<?php echo wp_kses_post( $settings['title'] ); ?>
				</span>
        	</h5>
        	<?php endif; ?>
			<?php if ( $settings['items'] ) : ?>
			<div class="swiper-container tst-footer-gallery">
                <div class="swiper-wrapper tst-cursor-zoom">
                  <?php foreach ( $settings['items'] as $index => $item ) : ?>
                  <div class="swiper-slide">
                    <div class="tst-footer-gal-item">
                      <?php if ( $item['image'] ) : $image = wp_get_attachment_image_url( $item['image']['id'], 'tastyc_140x140' ); $image_full = wp_get_attachment_image_url( $item['image']['id'], 'tastyc_1920xAuto' ); ?>
                      <img src="<?php echo esc_url( $image ); ?>" alt="<?php echo esc_attr( $item['name'] ); ?>" />
                      <a href="<?php echo esc_url( $image_full ); ?>" data-elementor-lightbox-slideshow="gallery-grid-footer" data-elementor-lightbox-title="<?php echo esc_attr( $item['name'] ); ?>" data-size="1920x1080" class="tst-overlay">
                        <i class="fas fa-search-plus"></i>
                      </a>
                      <?php endif; ?>
                    </div>
                  </div>
                  <?php endforeach; ?>
                </div>
                <div class="tst-gallery-nav">
                  <?php if ( $settings['link_text'] ) : ?>
                  <a<?php if ( $settings['link'] ) : ?><?php if ( $settings['link']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $settings['link']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?> href="<?php echo esc_url( $settings['link']['url'] ); ?>"<?php endif; ?> class="tst-label tst-color tst-anima-link">
                  	<span <?php echo $this->get_render_attribute_string( 'link_text' ); ?>>
											<?php echo wp_kses_post( $settings['link_text'] ); ?>
										</span>
										<span class="screen-reader-text"><?php echo esc_html( $settings['link_text'] ) . esc_html( $settings['title'] ); ?></span>
                  </a>
                  <?php endif; ?>
                  <div class="tst-fg-nav">
                    <div class="tst-slider-btn tst-fg-prev"><i class="fas fa-arrow-left"></i></div>
                    <div class="tst-slider-btn tst-fg-next"><i class="fas fa-arrow-right"></i></div>
                  </div>
                </div>
            </div>
			<?php endif; ?>
		</div>

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>

		<div class="tst-mb-60">
			<# if ( settings.title ) { #>
        	<h5 class="tst-mb-30 tst-text-shadow">
        		<span {{{ view.getRenderAttributeString( 'title' ) }}}>
	    			{{{ settings.title }}}
	    		</span>
        	</h5>
        	<# } #>
			<# if ( settings.items ) { #>
			<div class="swiper-container tst-footer-gallery">
                <div class="swiper-wrapper tst-cursor-zoom">
                  <# _.each( settings.items, function( item, index ) { #>
                  <div class="swiper-slide">
                    <div class="tst-footer-gal-item">
                      <# if ( item.image ) { #>
                      <img src="{{{ item.image.url }}}" alt="{{{ item.name }}}" />
                      <a href="{{{ item.image.url }}}" data-elementor-lightbox-slideshow="gallery-grid-footer" data-elementor-lightbox-title="{{{ item.name }}}" data-size="1920x1080" class="tst-overlay">
                        <i class="fas fa-search-plus"></i>
                      </a>
                      <# } #>
                    </div>
                  </div>
                  <# }); #>
                </div>
                <div class="tst-gallery-nav">
                  <# if ( settings.link_text ) { #>
                  <a<# if ( settings.link ) { #><# if ( settings.link.is_external ) { #> target="_blank"<# } #><# if ( settings.link.nofollow ) { #> rel="nofollow"<# } #> href="{{{ settings.link.url }}}"<# } #> class="tst-label tst-color tst-anima-link">
                  	<span {{{ view.getRenderAttributeString( 'link_text' ) }}}>
	    				{{{ settings.link_text }}}
	    			</span>
                  </a>
                  <# } #>
                  <div class="tst-fg-nav">
                    <div class="tst-slider-btn tst-fg-prev"><i class="fas fa-arrow-left"></i></div>
                    <div class="tst-slider-btn tst-fg-next"><i class="fas fa-arrow-right"></i></div>
                  </div>
                </div>
            </div>
			<# } #>
		</div>

		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tastyc_Footer_Gallery_Widget() );
