<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Services Widget.
 *
 * @since 1.0
 */
class Tastyc_History_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-history';
	}

	public function get_title() {
		return esc_html__( 'History', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'       => esc_html__( 'Subtitle', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter subtitle', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Subtitle', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Title', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => [
					'h1'  => __( 'H1', 'tastyc-plugin' ),
					'h2' => __( 'H2', 'tastyc-plugin' ),
					'h3' => __( 'H3', 'tastyc-plugin' ),
					'h4' => __( 'H4', 'tastyc-plugin' ),
					'div' => __( 'DIV', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'description',
			[
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter description', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Description', 'tastyc-plugin' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image', [
				'label' => esc_html__( 'Image', 'tastyc-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

    $repeater->add_control(
			'label', [
				'label'       => esc_html__( 'Label', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter label', 'tastyc-plugin' ),
				'default'	=> esc_html__( 'Label', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'title', [
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'	=> esc_html__( 'Enter title', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'desc', [
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Enter description', 'tastyc-plugin' ),
				'default'	=> esc_html__( 'Description', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ label }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'     => esc_html__( 'Subtitle Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-suptitle' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
				'label'     => esc_html__( 'Subtitle Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-suptitle',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-title--h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-title--h',
			]
		);

		$this->add_control(
			'desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-title--desc' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-title--desc',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

    $this->add_control(
			'item_label_color',
			[
				'label'     => esc_html__( 'Label Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-timeline .tst-timeline-item .tst-year' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_label_typography',
				'label'     => esc_html__( 'Label Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-timeline .tst-timeline-item .tst-year',
			]
		);

		$this->add_control(
			'item_title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-timeline .tst-timeline-item .tst-tl-content .tst-tl-text-frame h4' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-timeline .tst-timeline-item .tst-tl-content .tst-tl-text-frame h4',
			]
		);

		$this->add_control(
			'item_desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-timeline .tst-timeline-item .tst-tl-content .tst-tl-text-frame .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-timeline .tst-timeline-item .tst-tl-content .tst-tl-text-frame .tst-text',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'subtitle', 'basic' );
		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'description', 'basic' );

		?>

    <!-- container -->
    <div class="container tst-p-60-60">

      <div class="row align-items-center flex-sm-row-reverse">
        <?php if ( $settings['title'] || $settings['subtitle'] || $settings['description'] ) : ?>
        <div class="col-lg-12">
          <!-- about text -->
          <div class="tst-mb-60 text-center">
            <?php if ( $settings['subtitle'] ) : ?>
            <div class="tst-suptitle tst-suptitle-center tst-mb-15">
              <span <?php echo $this->get_render_attribute_string( 'subtitle' ); ?>>
                  <?php echo wp_kses_post( $settings['subtitle'] ); ?>
              </span>
            </div>
            <?php endif; ?>
            <?php if ( $settings['title'] ) : ?>
            <<?php echo esc_attr( $settings['title_tag'] ); ?> class="tst-title--h tst-mb-30">
              <span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
                  <?php echo wp_kses_post( $settings['title'] ); ?>
              </span>
            </<?php echo esc_attr( $settings['title_tag'] ); ?>>
            <?php endif; ?>
            <?php if ( $settings['description'] ) : ?>
            <p class="tst-text tst-title--desc tst-mb-30">
              <span <?php echo $this->get_render_attribute_string( 'description' ); ?>>
                  <?php echo wp_kses_post( $settings['description'] ); ?>
              </span>
            </p>
            <?php endif; ?>
          </div>
          <!-- about text end -->
        </div>
        <?php endif; ?>

        <?php if ( $settings['items'] ) : ?>
        <div class="col-lg-12">

          <div class="tst-timeline">
            <?php foreach ( $settings['items'] as $index => $item ) :
      		  $item_title = $this->get_repeater_setting_key( 'title', 'items', $index );
      		  $this->add_inline_editing_attributes( $item_title, 'basic' );

            $item_label = $this->get_repeater_setting_key( 'label', 'items', $index );
      		  $this->add_inline_editing_attributes( $item_label, 'none' );

      		  $item_desc = $this->get_repeater_setting_key( 'desc', 'items', $index );
      		  $this->add_inline_editing_attributes( $item_desc, 'basic' );
      		  ?>
            <div class="tst-timeline-item tst-mb-30">
              <?php if ( $item['label'] ) : ?>
              <div class="tst-year tst-mb-15">
                <span <?php echo $this->get_render_attribute_string( $item_label ); ?>>
                  <?php echo wp_kses_post( $item['label'] ); ?>
                </span>
              </div>
              <?php endif; ?>
              <div class="tst-tl-content">
                <?php if ( $item['image'] ) : $image = wp_get_attachment_image_url( $item['image']['id'], 'tastyc_950xAuto' ); ?>
                <div class="tst-ilust">
                  <img src="<?php echo esc_url( $image ); ?>" alt="img" />
                </div>
                <?php endif; ?>
                <div class="tst-tl-text-frame">
                  <?php if ( $item['title'] ) : ?>
                  <h4 class="tst-mb-30">
                    <span <?php echo $this->get_render_attribute_string( $item_title ); ?>>
        	        		<?php echo wp_kses_post( $item['title'] ); ?>
        	        	</span>
                  </h4>
                  <?php endif; ?>
                  <?php if ( $item['desc'] ) : ?>
                  <div class="tst-text">
                  	<div <?php echo $this->get_render_attribute_string( $item_desc ); ?>>
        	        		<?php echo wp_kses_post( $item['desc'] ); ?>
        	        	</div>
                  </div>
                  <?php endif; ?>
                </div>
              </div>
            </div>
            <?php endforeach; ?>
          </div>

        </div>
        <?php endif; ?>
      </div>

    </div>
    <!-- container end -->

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>

		<#
		view.addInlineEditingAttributes( 'subtitle', 'basic' );
		view.addInlineEditingAttributes( 'title', 'basic' );
		view.addInlineEditingAttributes( 'description', 'basic' );
		#>

		<!-- container -->
    <div class="container tst-p-60-60">

      <div class="row align-items-center flex-sm-row-reverse">
        <# if ( settings.title || settings.subtitle || settings.description ) { #>
        <div class="col-lg-12">
          <!-- about text -->
          <div class="tst-mb-60 text-center">
            <# if ( settings.subtitle ) { #>
            <div class="tst-suptitle tst-suptitle-center tst-mb-15">
              <span {{{ view.getRenderAttributeString( 'subtitle' ) }}}>
                  {{{ settings.subtitle }}}
              </span>
            </div>
            <# } #>
            <# if ( settings.title ) { #>
            <{{{ settings.title_tag }}} class="tst-title--h tst-mb-30">
              <span {{{ view.getRenderAttributeString( 'title' ) }}}>
                  {{{ settings.title }}}
              </span>
            </{{{ settings.title_tag }}}>
            <# } #>
            <# if ( settings.description ) { #>
            <p class="tst-text tst-title--desc tst-mb-30">
              <span {{{ view.getRenderAttributeString( 'description' ) }}}>
                  {{{ settings.description }}}
              </span>
            </p>
            <# } #>
          </div>
          <!-- about text end -->
        </div>
        <# } #>

        <# if ( settings.items ) { #>
        <div class="col-lg-12">

          <div class="tst-timeline">
            <# _.each( settings.items, function( item, index ) {
      			var item_title = view.getRepeaterSettingKey( 'title', 'items', index );
      			view.addInlineEditingAttributes( item_title, 'basic' );

            var item_label = view.getRepeaterSettingKey( 'label', 'items', index );
      			view.addInlineEditingAttributes( item_label, 'none' );

      			var item_desc = view.getRepeaterSettingKey( 'desc', 'items', index );
      			view.addInlineEditingAttributes( item_desc, 'basic' );
      		  #>
            <div class="tst-timeline-item tst-mb-30">
              <# if ( item.label ) { #>
              <div class="tst-year tst-mb-15">
                <span {{{ view.getRenderAttributeString( item_label ) }}}>
                  {{{ item.label }}}
                </span>
              </div>
              <# } #>
              <div class="tst-tl-content">
                <# if ( item.image ) { #>
                <div class="tst-ilust">
                  <img src="{{{ item.image.url }}}" alt="img" />
                </div>
                <# } #>
                <div class="tst-tl-text-frame">
                  <# if ( item.title ) { #>
                  <h4 class="tst-mb-30">
                    <span {{{ view.getRenderAttributeString( item_title ) }}}>
            					{{{ item.title }}}
            				</span>
                  </h4>
                  <# } #>
                  <# if ( item.desc ) { #>
                  <div class="tst-text">
                    <div {{{ view.getRenderAttributeString( item_desc ) }}}>
            					{{{ item.desc }}}
            				</div>
                  </div>
                  <# } #>
                </div>
              </div>
            </div>
            <# }); #>
          </div>

        </div>
        <# } #>
      </div>

    </div>
		<!-- container end -->

		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tastyc_History_Widget() );
