<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Intro Widget.
 *
 * @since 1.0
 */
class Tastyc_Intro_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-intro';
	}

	public function get_title() {
		return esc_html__( 'Intro', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title_value',
			[
				'label'       => esc_html__( 'Title Value', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'dynamic',
				'options' => [
					'dynamic'  => __( 'Dynamic', 'tastyc-plugin' ),
					'static' => __( 'Static', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'subtitle_value',
			[
				'label'       => esc_html__( 'Subtitle Value', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'dynamic',
				'options' => [
					'dynamic'  => __( 'Dynamic', 'tastyc-plugin' ),
					'static' => __( 'Static', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'description_value',
			[
				'label'       => esc_html__( 'Description Value', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'dynamic',
				'options' => [
					'dynamic'  => __( 'Dynamic', 'tastyc-plugin' ),
					'static' => __( 'Static', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'       => esc_html__( 'Subtitle', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter subtitle', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Subtitle', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Title', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h1',
				'options' => [
					'h1'  => __( 'H1', 'tastyc-plugin' ),
					'h2' => __( 'H2', 'tastyc-plugin' ),
					'h3' => __( 'H3', 'tastyc-plugin' ),
					'div' => __( 'DIV', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'description',
			[
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter description', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Description', 'tastyc-plugin' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'breadcrumbs_tab',
			[
				'label' => esc_html__( 'Breadcrumbs', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'breadcrumbs',
			[
				'label'       => esc_html__( 'Breadcrumbs', 'tastyc-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Yes', 'tastyc-plugin' ),
				'label_off' => __( 'No', 'tastyc-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'background_tab',
			[
				'label' => esc_html__( 'Background', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'image',
			[
				'label'       => esc_html__( 'Background Image', 'tastyc-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'     => esc_html__( 'Subtitle Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-banner .tst-suptitle' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
				'label'     => esc_html__( 'Subtitle Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-banner .tst-suptitle',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-banner .tst-title--h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-banner .tst-title--h',
			]
		);

		$this->add_control(
			'desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-banner .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-banner .tst-text',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'breadcrumbs_styling',
			[
				'label'     => esc_html__( 'Breadcrumbs', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'breadcrumbs_bgcolor',
			[
				'label'     => esc_html__( 'Background Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-breadcrumbs' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'breadcrumbs_link_color',
			[
				'label'     => esc_html__( 'Background Link Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .tst-breadcrumbs li a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'breadcrumbs_link_typography',
				'label'     => esc_html__( 'Background Link Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .tst-breadcrumbs li a',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'subtitle', 'basic' );
		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'description', 'basic' );

		$page_id = get_queried_object_id();

		if ( $page_id ) {
			$intro_title = get_field( 'intro_title', $page_id );
			if ( !$intro_title ) {
				$intro_title = get_the_title( $page_id );
			}

			$intro_sub = get_field( 'intro_subtitle', $page_id );
			$intro_text = get_field( 'intro_description', $page_id );

			if ( $intro_title && $settings['title_value'] == 'dynamic'  ) {
				$settings['title'] = $intro_title;
			}
			if ( $intro_sub && $settings['subtitle_value'] == 'dynamic' ) {
				$settings['subtitle'] = $intro_sub;
			}
			if ( $intro_text && $settings['description_value'] == 'dynamic' ) {
				$settings['description'] = $intro_text;
			}
		}

		?>

		<!-- banner -->
		<div class="tst-banner tst-small-banner">
			<div class="tst-cover-frame">
			  <?php if ( $settings['image'] ) : $image = wp_get_attachment_image_url( $settings['image']['id'], 'tastyc_1920xAuto' ); ?>
			  <img src="<?php echo esc_url( $image ); ?>" alt="" class="tst-cover tst-parallax">
			  <?php endif; ?>
			  <div class="tst-overlay"></div>
			</div>
			<div class="tst-banner-content-frame">
			  <div class="container">
			    <div class="tst-main-title-frame">
			      <div class="tst-main-title text-center">
			      	<?php if ( $settings['subtitle'] ) : ?>
			        <div class="tst-suptitle tst-suptitle-center tst-suptitle-mobile-center tst-text-shadow tst-white-2 tst-mb-15">
			        	<span <?php echo $this->get_render_attribute_string( 'subtitle' ); ?>>
				          	<?php echo wp_kses_post( $settings['subtitle'] ); ?>
				        </span>
			        </div>
			        <?php endif; ?>
			        <?php if ( $settings['title'] ) : ?>
			        <<?php echo esc_attr( $settings['title_tag'] ); ?> class="tst-title--h tst-white-2 tst-text-shadow tst-mb-30">
			        	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
				          	<?php echo wp_kses_post( $settings['title'] ); ?>
				        </span>
			    	</<?php echo esc_attr( $settings['title_tag'] ); ?>>
			        <?php endif; ?>
			        <?php if ( $settings['description'] ) : ?>
			        <div class="tst-text tst-text-shadow tst-text-lg tst-white-2 tst-mb-30">
			        	<span <?php echo $this->get_render_attribute_string( 'description' ); ?>>
				          	<?php echo wp_kses_post( $settings['description'] ); ?>
				        </span>
			        </div>
			        <?php endif; ?>
			        <?php if ( $settings['breadcrumbs'] == 'yes' ) : tastyc_breadcrumbs( $page_id ); endif; ?>
			      </div>
			    </div>
			  </div>
			</div>
		</div>
		<!-- banner end -->

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>

		<#
		view.addInlineEditingAttributes( 'subtitle', 'basic' );
		view.addInlineEditingAttributes( 'title', 'basic' );
		view.addInlineEditingAttributes( 'description', 'basic' );
		#>

		<!-- banner -->
		<div class="tst-banner tst-small-banner">
			<div class="tst-cover-frame">
			  <# if ( settings.image ) { #>
			  <img src="{{{ settings.image.url }}}" alt="" class="tst-cover tst-parallax">
			  <# } #>
			  <div class="tst-overlay"></div>
			</div>
			<div class="tst-banner-content-frame">
			  <div class="container">
			    <div class="tst-main-title-frame">
			      <div class="tst-main-title text-center">
			      	<# if ( settings.subtitle ) { #>
			        <div class="tst-suptitle tst-suptitle-center tst-suptitle-mobile-center tst-text-shadow tst-white-2 tst-mb-15">
			        	<span {{{ view.getRenderAttributeString( 'subtitle' ) }}}>
				          	{{{ settings.subtitle }}}
				        </span>
			        </div>
			        <# } #>
			        <# if ( settings.title ) { #>
			        <{{{ settings.title_tag }}} class="tst-title--h tst-white-2 tst-text-shadow tst-mb-30">
			        	<span {{{ view.getRenderAttributeString( 'title' ) }}}>
				          	{{{ settings.title }}}
				        </span>
			    	</{{{ settings.title_tag }}}>
			        <# } #>
			        <# if ( settings.description ) { #>
			        <div class="tst-text tst-text-shadow tst-text-lg tst-white-2 tst-mb-30">
			        	<span {{{ view.getRenderAttributeString( 'description' ) }}}>
				          	{{{ settings.description }}}
				        </span>
			        </div>
			        <# } #>
			        <# if ( settings.breadcrumbs == 'yes' ) { #>
			        <ul class="tst-breadcrumbs">
			          <li><a href="#." class="tst-anima-link"><?php echo esc_html__( 'Home', 'tastyc-plugin' ); ?></a></li>
			          <li class="tst-active"><a href="#."><?php echo esc_html__( 'Current Page', 'tastyc-plugin' ); ?></a></li>
			        </ul>
			        <# } #>
			      </div>
			    </div>
			  </div>
			</div>
		</div>
		<!-- banner end -->

		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tastyc_Intro_Widget() );
