<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Tastyc Pricing Widget.
 *
 * @since 1.0
 */
class Tastyc_Pricing_Widget extends Widget_Base {

	public function get_name() {
		return 'tastyc-pricing';
	}

	public function get_title() {
		return esc_html__( 'Pricing', 'tastyc-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'tastyc-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'       => esc_html__( 'Subtitle', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter subtitle', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Subtitle', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Title', 'tastyc-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'tastyc-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => [
					'h1'  => __( 'H1', 'tastyc-plugin' ),
					'h2' => __( 'H2', 'tastyc-plugin' ),
					'h3' => __( 'H3', 'tastyc-plugin' ),
					'h4' => __( 'H4', 'tastyc-plugin' ),
					'div' => __( 'DIV', 'tastyc-plugin' ),
				],
			]
		);

		$this->add_control(
			'description',
			[
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter description', 'tastyc-plugin' ),
				'default'     => esc_html__( 'Description', 'tastyc-plugin' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'marked',
			[
				'label'       => esc_html__( 'Checked Yes/No', 'tastyc-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Yes', 'tastyc-plugin' ),
				'label_off' => __( 'No', 'tastyc-plugin' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$repeater->add_control(
			'icon', [
				'label' => esc_html__( 'Icon', 'tastyc-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'title', [
				'label'       => esc_html__( 'Title', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'tastyc-plugin' ),
				'default'	=> esc_html__( 'Enter title', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'price_pre', [
				'label'       => esc_html__( 'Price Prefix', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter prefix', 'tastyc-plugin' ),
				'default'	=> esc_html__( 'Enter prefix', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'price_val', [
				'label'       => esc_html__( 'Price Value', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter value', 'tastyc-plugin' ),
				'default'	=> esc_html__( 'Enter value', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'price_after', [
				'label'       => esc_html__( 'Price After', 'tastyc-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter after', 'tastyc-plugin' ),
				'default'	=> esc_html__( 'Enter after', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'desc', [
				'label'       => esc_html__( 'Description', 'tastyc-plugin' ),
				'type'        => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Enter description', 'tastyc-plugin' ),
				'default'	=> esc_html__( 'Description', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'button1',
			[
				'label'       => esc_html__( 'Button 1 (label)', 'tastyc-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Button', 'tastyc-plugin' ),
			]
		);

		$repeater->add_control(
			'link1',
			[
				'label'       => esc_html__( 'Button 1 (link)', 'tastyc-plugin' ),
				'type' => Controls_Manager::URL,
				'show_external' => true,
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'tastyc-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ title }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'     => esc_html__( 'Subtitle Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-suptitle' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
				'label'     => esc_html__( 'Subtitle Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-suptitle',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-title--h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-title--h',
			]
		);

		$this->add_control(
			'desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-title--desc' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-title--desc',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'tastyc-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_title_color',
			[
				'label'     => esc_html__( 'Title Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-pricing-box h5' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_title_typography',
				'label'     => esc_html__( 'Title Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-pricing-box h5',
			]
		);

		$this->add_control(
			'item_price_color',
			[
				'label'     => esc_html__( 'Price Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-pricing-box .tst-hours-price' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_price_typography',
				'label'     => esc_html__( 'Price Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-pricing-box .tst-hours-price',
			]
		);

		$this->add_control(
			'item_desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-pricing-box .tst-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_desc_typography',
				'label'     => esc_html__( 'Description Typography', 'tastyc-plugin' ),
				'selector' => '{{WRAPPER}} .row .tst-pricing-box .tst-text',
			]
		);

		$this->add_control(
			'btn_color',
			[
				'label'     => esc_html__( 'Button Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-pricing-box .tst-btn' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'btn_bgcolor',
			[
				'label'     => esc_html__( 'Button BG Color', 'tastyc-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .row .tst-pricing-box .tst-btn' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label'     => esc_html__( 'Button Typography', 'tastyc-plugin' ),
				'name'     => 'btn_typography',
				'selector' => '{{WRAPPER}} .row .tst-pricing-box .tst-btn',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'subtitle', 'basic' );
		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'description', 'basic' );

		?>

		<!-- container -->
		<div class="container">

		<div class="row">
		  <?php if ( $settings['title'] || $settings['subtitle'] || $settings['description'] ) : ?>
          <div class="col-lg-12">

            <!-- title -->
            <div class="text-center">
              <?php if ( $settings['subtitle'] ) : ?>
              <div class="tst-suptitle tst-suptitle-center tst-mb-15">
              	<span <?php echo $this->get_render_attribute_string( 'subtitle' ); ?>>
		          	<?php echo wp_kses_post( $settings['subtitle'] ); ?>
		        </span>
              </div>
              <?php endif; ?>
              <?php if ( $settings['title'] ) : ?>
              <<?php echo esc_attr( $settings['title_tag'] ); ?> class="tst-title--h tst-mb-30">
              	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
		          	<?php echo wp_kses_post( $settings['title'] ); ?>
		        </span>
              </<?php echo esc_attr( $settings['title_tag'] ); ?>>
              <?php endif; ?>
              <?php if ( $settings['description'] ) : ?>
              <p class="tst-text tst-title--desc tst-mb-60">
              	<span <?php echo $this->get_render_attribute_string( 'description' ); ?>>
		          	<?php echo wp_kses_post( $settings['description'] ); ?>
		        </span>
              </p>
              <?php endif; ?>
            </div>
            <!-- title end -->

          </div>
          <?php endif; ?>

          <?php if ( $settings['items'] ) : ?>
		  <?php foreach ( $settings['items'] as $index => $item ) :
		  $item_title = $this->get_repeater_setting_key( 'title', 'items', $index );
		  $this->add_inline_editing_attributes( $item_title, 'basic' );

		  $item_desc = $this->get_repeater_setting_key( 'desc', 'items', $index );
		  $this->add_inline_editing_attributes( $item_desc, 'basic' );

		  $item_button = $this->get_repeater_setting_key( 'button1', 'items', $index );
		  $this->add_inline_editing_attributes( $item_button, 'none' );

		  $item_price_pre = $this->get_repeater_setting_key( 'price_pre', 'items', $index );
		  $this->add_inline_editing_attributes( $item_price_pre, 'basic' );

		  $item_price_val = $this->get_repeater_setting_key( 'price_val', 'items', $index );
		  $this->add_inline_editing_attributes( $item_price_val, 'basic' );

		  $item_price_after = $this->get_repeater_setting_key( 'price_after', 'items', $index );
		  $this->add_inline_editing_attributes( $item_price_after, 'basic' );
		  ?>
          <div class="col-lg-4">

            <!-- pricing box -->
            <div class="tst-pricing-box tst-mb-60 <?php if ( $item['marked'] == 'yes' ) : ?>marked<?php endif; ?>">
              <?php if ( $item['icon'] ) : $image = wp_get_attachment_image_url( $item['icon']['id'], 'tastyc_256x256' ); ?>
              <img src="<?php echo esc_url( $image ); ?>" alt="" class="tst-mb-15" />
              <?php endif; ?>
              <?php if ( $item['title'] ) : ?>
              <h5 class="tst-mb-30">
              	<span <?php echo $this->get_render_attribute_string( $item_title ); ?>>
	        		<?php echo wp_kses_post( $item['title'] ); ?>
	        	</span>
              </h5>
              <?php endif; ?>
              <?php if ( $item['price_val'] || $item['price_pre'] || $item['price_after'] ) : ?>
              <div class="tst-hours-price">
              	<span class="tst-price-pre">
              		<span <?php echo $this->get_render_attribute_string( $item_price_pre ); ?>><?php echo wp_kses_post( $item['price_pre'] ); ?></span>
              	</span>
              	<span class="tst-price-val">
              		<span <?php echo $this->get_render_attribute_string( $item_price_val ); ?>><?php echo wp_kses_post( $item['price_val'] ); ?></span>
              	</span>
				<span class="tst-price-after">
					<span <?php echo $this->get_render_attribute_string( $item_price_after ); ?>><?php echo wp_kses_post( $item['price_after'] ); ?></span>
				</span>
              </div>
              <?php endif; ?>
              <?php if ( $item['desc'] ) : ?>
              <div class="tst-text">
              	<div <?php echo $this->get_render_attribute_string( $item_desc ); ?>>
	        		<?php echo wp_kses_post( $item['desc'] ); ?>
	        	</div>
              </div>
              <?php endif; ?>
              <?php if ( $item['button1'] ) : ?>
              <a<?php if ( $item['link1'] ) : if ( $item['link1']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $item['link1']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?> href="<?php echo esc_url( $item['link1']['url'] ); ?>"<?php endif; ?> class="tst-btn tst-btn-shadow tst-mt-30">
              	<span <?php echo $this->get_render_attribute_string( 'button1' ); ?>>
					<?php echo wp_kses_post( $item['button1'] ); ?>
				</span>
              </a>
              <?php endif; ?>
            </div>
            <!-- pricing box end -->

          </div>
          <?php endforeach; ?>
          <?php endif; ?>

        </div>

        <div class="tst-spacer tst-spacer-only-bottom-space"></div>

		</div>
		<!-- container end -->

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>

		<#
		view.addInlineEditingAttributes( 'subtitle', 'basic' );
		view.addInlineEditingAttributes( 'title', 'basic' );
		view.addInlineEditingAttributes( 'description', 'basic' );
		#>

		<!-- container -->
		<div class="container">

		<div class="row">
		  <# if ( settings.title || settings.subtitle || settings.description ) { #>
          <div class="col-lg-12">

            <!-- title -->
            <div class="text-center">
              <# if ( settings.subtitle ) { #>
              <div class="tst-suptitle tst-suptitle-center tst-mb-15">
              	<span {{{ view.getRenderAttributeString( 'subtitle' ) }}}>
		          	{{{ settings.subtitle }}}
		        </span>
              </div>
              <# } #>
              <# if ( settings.title ) { #>
              <{{{ settings.title_tag }}} class="tst-title--h tst-mb-30">
              	<span {{{ view.getRenderAttributeString( 'title' ) }}}>
		          	{{{ settings.title }}}
		        </span>
              </{{{ settings.title_tag }}}>
              <# } #>
              <# if ( settings.description ) { #>
              <p class="tst-text tst-title--desc tst-mb-60">
              	<span {{{ view.getRenderAttributeString( 'description' ) }}}>
		          	{{{ settings.description }}}
		        </span>
              </p>
              <# } #>
            </div>
            <!-- title end -->

          </div>
          <# } #>

          <# if ( settings.items ) { #>
		  <# _.each( settings.items, function( item, index ) {
			var item_title = view.getRepeaterSettingKey( 'title', 'items', index );
			view.addInlineEditingAttributes( item_title, 'basic' );

			var item_desc = view.getRepeaterSettingKey( 'desc', 'items', index );
			view.addInlineEditingAttributes( item_desc, 'basic' );

			var item_price_pre = view.getRepeaterSettingKey( 'price_pre', 'items', index );
			view.addInlineEditingAttributes( item_price_pre, 'basic' );

			var item_price_val = view.getRepeaterSettingKey( 'price_val', 'items', index );
			view.addInlineEditingAttributes( item_price_val, 'basic' );

			var item_price_after = view.getRepeaterSettingKey( 'price_after', 'items', index );
			view.addInlineEditingAttributes( item_price_after, 'basic' );

			var item_button = view.getRepeaterSettingKey( 'button1', 'items', index );
			view.addInlineEditingAttributes( item_button, 'basic' );
		  #>
          <div class="col-lg-4">

            <!-- pricing box -->
            <div class="tst-pricing-box tst-mb-60 <# if ( item.marked == 'yes' ) { #>marked<# } #>">
              <# if ( item.icon ) { #>
              <img src="{{{ item.icon.url }}}" alt="" class="tst-mb-15" />
              <# } #>
              <# if ( item.title ) { #>
              <h5 class="tst-mb-30">
              	<span {{{ view.getRenderAttributeString( item_title ) }}}>
					{{{ item.title }}}
				</span>
              </h5>
              <# } #>
              <# if ( item.price_pre || item.price_val || item.price_after ) { #>
              <div class="tst-hours-price">
              	<span class="tst-price-pre">
              		<span {{{ view.getRenderAttributeString( item_price_pre ) }}}>{{{ item.price_pre }}}</span>
              	</span>
              	<span class="tst-price-val">
              		<span {{{ view.getRenderAttributeString( item_price_val ) }}}>{{{ item.price_val }}}</span>
              	</span>
				<span class="tst-price-after">
					<span {{{ view.getRenderAttributeString( item_price_after ) }}}>{{{ item.price_after }}}</span>
				</span>
              </div>
              <# } #>
              <# if ( item.desc ) { #>
              <div class="tst-text">
              	<div {{{ view.getRenderAttributeString( item_desc ) }}}>
					{{{ item.desc }}}
				</div>
              </div>
              <# } #>
              <# if ( item.button1 ) { #>
              <a<# if ( item.link1 ) { if ( item.link1.is_external ) { #> target="_blank"<# } #><# if ( item.link1.nofollow ) { #> rel="nofollow"<# } #> href="{{{ item.link1.url }}}"<# } #> class="tst-btn tst-btn-shadow tst-mt-30">
              	<span {{{ view.getRenderAttributeString( item_button ) }}}>
					{{{ item.button1 }}}
				</span>
              </a>
              <# } #>
            </div>
            <!-- pricing box end -->

          </div>
          <# }); #>
          <# } #>

        </div>

        <div class="tst-spacer tst-spacer-only-bottom-space"></div>

		</div>
		<!-- container end -->

		<?php
	}
}

Plugin::instance()->widgets_manager->register( new Tastyc_Pricing_Widget() );
